// Cloudflare Worker for Cross-Border Translator
// Handles API requests to Qwen-MT securely

export default {
  async fetch(request, env, ctx) {
    // 1. 处理 CORS 预检请求 (OPTIONS)
    if (request.method === "OPTIONS") {
      return new Response(null, {
        headers: {
          "Access-Control-Allow-Origin": "*", // 允许所有域名（开发测试方便，生产环境建议限制为插件 ID）
          "Access-Control-Allow-Methods": "POST, OPTIONS",
          "Access-Control-Allow-Headers": "Content-Type",
        },
      });
    }

    // 2. 仅允许 POST 请求
    if (request.method !== "POST") {
      return new Response("Method Not Allowed", { status: 405 });
    }

    try {
      // 3. 解析请求体
      const { text, source_lang = "auto", target_lang = "en" } = await request.json();

      if (!text) {
        return new Response(JSON.stringify({ error: "Text is required" }), {
          status: 400,
          headers: { "Content-Type": "application/json" },
        });
      }

      // 4. 构造 Qwen-MT 请求
      const API_KEY = env.DASHSCOPE_API_KEY; // 从环境变量获取 Key
      if (!API_KEY) {
        throw new Error("API Key not configured on server");
      }

      const url = "https://dashscope.aliyuncs.com/compatible-mode/v1/chat/completions";
      
      // 跨境电商领域 Prompt (参考 PROJECT.md)
      const domainPrompt = "The text is from cross-border e-commerce domain, including product descriptions, marketing content, customer service dialogues, and logistics information. Pay attention to e-commerce terminology, product attributes, and customer-friendly tone.";

      const payload = {
        model: "qwen-mt-flash",
        messages: [
          {
            role: "user",
            content: text
          }
        ],
        translation_options: {
          source_lang: source_lang,
          target_lang: target_lang,
          domains: domainPrompt
        }
      };

      // 5. 调用阿里云 API
      const response = await fetch(url, {
        method: "POST",
        headers: {
          "Authorization": `Bearer ${API_KEY}`,
          "Content-Type": "application/json"
        },
        body: JSON.stringify(payload)
      });

      const data = await response.json();

      // 6. 错误处理与响应转发
      if (!response.ok) {
        return new Response(JSON.stringify(data), {
          status: response.status,
          headers: {
            "Content-Type": "application/json",
            "Access-Control-Allow-Origin": "*"
          }
        });
      }

      // 7. 返回成功结果
      // 提取核心翻译内容，简化前端处理
      const translatedText = data.choices?.[0]?.message?.content || "";

      return new Response(JSON.stringify({ translation: translatedText }), {
        headers: {
          "Content-Type": "application/json",
          "Access-Control-Allow-Origin": "*"
        }
      });

    } catch (error) {
      return new Response(JSON.stringify({ error: error.message }), {
        status: 500,
        headers: {
          "Content-Type": "application/json",
          "Access-Control-Allow-Origin": "*"
        }
      });
    }
  },
};
