// 跨境翻译宝 Content Script
  // console.log('[跨境翻译宝] 🚀 新版 Content Script 已加载！(版本号: v2.0-FixDrag)'); // 发布时注释掉调试日志

  (function() {
    // --- 调试辅助 ---
    // 如果您看到了这个 log，说明代码已更新
    
    // 防止重复注入 - 强制清理旧的 root
    // 这一次我们更彻底一点，清理所有可能的残留
    document.querySelectorAll('#kbt-root').forEach(el => el.remove());

    // --- 配置 ---
    // 引入外部配置文件 (通过 Fetch 或 简单变量，这里为了兼容非模块化环境，我们手动定义)
    // 注意：Content Script 默认不是 Module，import 需要 manifest 配置。
    // 为了稳妥，我们在这里定义一个全局配置对象，未来发布新版只需改这里。
    
    const API_CONFIG = {
      // [生产环境] 请将此域名修改为您购买的自定义域名，例如 https://api.kuajing-fanyi.com
      // 目前暂时使用 Vercel 域名
      BASE_URL: 'https://kuajing-fanyi.vercel.app',
      PATH: '/api/translate'
    };

    const WORKER_URL = `${API_CONFIG.BASE_URL}${API_CONFIG.PATH}`;
    
    const MAX_BATCH_SIZE = 4500; // Qwen-MT 建议单次请求限制

  // 92 种支持语言列表
  const SUPPORTED_LANGUAGES = [
    { name: "English", code: "en" },
    { name: "简体中文", code: "zh" },
    { name: "Traditional Chinese", code: "zh_tw" },
    { name: "Russian", code: "ru" },
    { name: "Japanese", code: "ja" },
    { name: "Korean", code: "ko" },
    { name: "Spanish", code: "es" },
    { name: "French", code: "fr" },
    { name: "Portuguese", code: "pt" },
    { name: "German", code: "de" },
    { name: "Italian", code: "it" },
    { name: "Thai", code: "th" },
    { name: "Vietnamese", code: "vi" },
    { name: "Indonesian", code: "id" },
    { name: "Malay", code: "ms" },
    { name: "Arabic", code: "ar" },
    { name: "Hindi", code: "hi" },
    { name: "Hebrew", code: "he" },
    { name: "Burmese", code: "my" },
    { name: "Tamil", code: "ta" },
    { name: "Urdu", code: "ur" },
    { name: "Bengali", code: "bn" },
    { name: "Polish", code: "pl" },
    { name: "Dutch", code: "nl" },
    { name: "Romanian", code: "ro" },
    { name: "Turkish", code: "tr" },
    { name: "Khmer", code: "km" },
    { name: "Lao", code: "lo" },
    { name: "Cantonese", code: "yue" },
    { name: "Czech", code: "cs" },
    { name: "Greek", code: "el" },
    { name: "Swedish", code: "sv" },
    { name: "Hungarian", code: "hu" },
    { name: "Danish", code: "da" },
    { name: "Finnish", code: "fi" },
    { name: "Ukrainian", code: "uk" },
    { name: "Bulgarian", code: "bg" },
    { name: "Serbian", code: "sr" },
    { name: "Telugu", code: "te" },
    { name: "Afrikaans", code: "af" },
    { name: "Armenian", code: "hy" },
    { name: "Assamese", code: "as" },
    { name: "Asturian", code: "ast" },
    { name: "Basque", code: "eu" },
    { name: "Belarusian", code: "be" },
    { name: "Bosnian", code: "bs" },
    { name: "Catalan", code: "ca" },
    { name: "Cebuano", code: "ceb" },
    { name: "Croatian", code: "hr" },
    { name: "Egyptian Arabic", code: "arz" },
    { name: "Estonian", code: "et" },
    { name: "Galician", code: "gl" },
    { name: "Georgian", code: "ka" },
    { name: "Gujarati", code: "gu" },
    { name: "Haitian Creole", code: "ht" },
    { name: "Hausa", code: "ha" },
    { name: "Hawaiian", code: "haw" },
    { name: "Icelandic", code: "is" },
    { name: "Igbo", code: "ig" },
    { name: "Irish", code: "ga" },
    { name: "Javanese", code: "jv" },
    { name: "Kannada", code: "kn" },
    { name: "Kazakh", code: "kk" },
    { name: "Kyrgyz", code: "ky" },
    { name: "Luxembourgish", code: "lb" },
    { name: "Malagasy", code: "mg" },
    { name: "Malayalam", code: "ml" },
    { name: "Maltese", code: "mt" },
    { name: "Maori", code: "mi" },
    { name: "Marathi", code: "mr" },
    { name: "Mongolian", code: "mn" },
    { name: "Nepali", code: "ne" },
    { name: "Norwegian", code: "no" },
    { name: "Oriya", code: "or" },
    { name: "Pashto", code: "ps" },
    { name: "Punjabi", code: "pa" },
    { name: "Quechua", code: "qu" },
    { name: "Samoan", code: "sm" },
    { name: "Scottish Gaelic", code: "gd" },
    { name: "Shona", code: "sn" },
    { name: "Sindhi", code: "sd" },
    { name: "Sinhala", code: "si" },
    { name: "Slovak", code: "sk" },
    { name: "Slovenian", code: "sl" },
    { name: "Somali", code: "so" },
    { name: "Tajik", code: "tg" },
    { name: "Turkmen", code: "tk" },
    { name: "Uzbek", code: "uz" },
    { name: "Welsh", code: "cy" },
    { name: "Xhosa", code: "xh" },
    { name: "Yiddish", code: "yi" },
    { name: "Yoruba", code: "yo" },
    { name: "Zulu", code: "zu" }
  ];

  // --- 状态管理 ---
  let isTranslating = false;
  let isTranslated = false;
  let abortController = null;
  let targetLang = 'zh'; // 默认目标语言
  
  // 缓存与记录
  let processedNodes = new WeakSet();
  // 用于记录短语翻译的原始文本，以便还原 Node -> originalText
  const modifiedTextNodes = new Map();
  // 用于记录 Inline Loader 的映射 Node -> LoaderElement
  const inlineLoaders = new Map();

  // --- 初始化存储 ---
  chrome.storage.local.get(['targetLang'], (result) => {
    if (result.targetLang) {
      targetLang = result.targetLang;
    }
  });

  // --- UI 初始化 ---
  
  const host = document.createElement('div');
  host.id = 'kbt-root';
  document.body.appendChild(host);

  const shadow = host.attachShadow({ mode: 'open' });

  // 引入全局样式 (解决 Light DOM 中 inline-loader 样式问题)
  const globalStyle = document.createElement('style');
  globalStyle.id = 'kbt-global-style';
  globalStyle.textContent = `
    .kbt-inline-loader {
      display: inline-block;
      width: 12px;
      height: 12px;
      border: 2px solid #ccc; /* 浅灰色 */
      border-bottom-color: transparent;
      border-radius: 50%;
      animation: kbt-rotation 1s linear infinite;
      margin-left: 5px;
      vertical-align: middle;
      position: relative;
      z-index: 9999;
    }
    
    @keyframes kbt-rotation {
      0% { transform: rotate(0deg); }
      100% { transform: rotate(360deg); }
    }
  `;
  document.head.appendChild(globalStyle);

  // 引入样式 (Shadow DOM)
  const style = document.createElement('style');
  style.textContent = `
    /* 容器：右侧吸附，垂直排列 */
    .kbt-container {
      position: fixed !important;
      top: 50%;
      right: 0; /* 强制贴紧右侧 */
      /* 移除 bottom: auto, left: auto 的 !important，允许 JS 覆盖 */
      transform: translateY(-50%);
      display: flex !important;
      flex-direction: column !important;
      gap: 15px !important;
      z-index: 2147483647 !important;
      pointer-events: auto !important;
      margin: 0 !important;
      padding: 0 !important;
      width: auto !important;
      height: auto !important;
      background: transparent !important;
      border: none !important;
      box-shadow: none !important;
    }

    /* 按钮通用样式 */
    .kbt-icon-btn {
      width: 50px !important;
      height: 50px !important;
      cursor: pointer !important;
      position: relative !important;
      display: flex !important;
      align-items: center !important;
      justify-content: center !important;
      margin: 0 !important;
      padding: 0 !important;
      background: transparent !important;
      border: none !important;
      box-shadow: none !important;
      outline: none !important;
      transition: all 0.3s ease !important; /* 添加平滑过渡 */
    }

    /* 设置按钮容器特殊样式 - 用于滑动效果 */
    #kbt-set-btn-container {
      position: absolute;
      top: 65px; /* 在翻译按钮下方 15px + 50px */
      right: 0;
      width: 50px;
      height: 50px;
      overflow: hidden; /* 隐藏滑动出去的部分 */
      pointer-events: none; /* 默认不阻挡鼠标，除非显示 */
    }

    #kbt-set-btn {
      position: absolute;
      right: -60px; /* 初始隐藏在右侧 */
      transition: right 0.3s cubic-bezier(0.4, 0.0, 0.2, 1);
      pointer-events: auto;
    }

    /* 当激活时滑入 */
    .kbt-set-visible #kbt-set-btn {
      right: 0;
    }

    .kbt-icon-btn img {
      width: 100% !important;
      height: 100% !important;
      object-fit: contain !important;
      display: block !important;
      margin: 0 !important;
      padding: 0 !important;
    }

    /* 隐藏类 */
    .kbt-hidden {
      display: none !important;
      visibility: hidden !important;
      opacity: 0 !important;
      pointer-events: none !important;
    }

    /* 用户提供的 Loader CSS - 重构修复层级问题 */
    .loader {
      /* 父元素作为背景容器 */
      width: 50px;
      height: 50px;
      background-color: white;
      border-radius: 50%;
      box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
      
      /* 绝对定位以填满父容器，确保居中 */
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      margin: auto;
      
      /* Flex 布局让子元素(spinner)居中 */
      display: flex;
      align-items: center;
      justify-content: center;
    }

    .loader::after {
      /* 子伪元素作为旋转 Spinner */
      content: '';
      width: 30px; 
      height: 30px; 
      border: 4px solid; 
      border-color: #FF3D00 transparent; /* 主题色修改为 #FF3D00 */
      border-radius: 50%; 
      box-sizing: border-box; 
      animation: rotation 1s linear infinite; 
    }

    @keyframes rotation { 
      0% { 
        transform: rotate(0deg); 
      } 
      100% { 
        transform: rotate(360deg); 
      } 
    }

    /* --- 设置面板样式 --- */

    .kbt-settings-panel {
      position: absolute;
      top: 50%;
      right: 60px; /* 位于图标左侧 (50px icon + 10px gap) */
      transform: translateY(-50%);
      width: 320px;
      background: white;
      border-radius: 12px;
      box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
      overflow: hidden;
      border: 1px solid #eee;
      animation: slideIn 0.2s ease-out;
    }

    @keyframes slideIn {
      from { opacity: 0; transform: translate(10px, -50%); }
      to { opacity: 1; transform: translate(0, -50%); }
    }

    .kbt-settings-header {
      background: linear-gradient(135deg, #2ecc71, #27ae60);
      padding: 15px;
      color: white;
      font-weight: 600;
      display: flex;
      align-items: center;
      gap: 10px;
      font-size: 16px;
    }

    .kbt-header-icon {
      width: 24px;
      height: 24px;
      object-fit: contain;
    }

    .kbt-settings-body {
      padding: 20px;
      display: flex;
      flex-direction: column;
      gap: 20px;
    }

    .kbt-form-group {
      display: flex;
      flex-direction: column;
      gap: 8px;
    }

    .kbt-form-group label {
      font-size: 14px;
      color: #666;
      font-weight: 500;
    }

    .kbt-fake-select {
      padding: 10px 12px;
      background: #f5f5f5;
      border: 1px solid #ddd;
      border-radius: 6px;
      color: #999;
      font-size: 14px;
      cursor: not-allowed;
    }

    #kbt-lang-select {
      padding: 10px 12px;
      background: white;
      border: 1px solid #ddd;
      border-radius: 6px;
      font-size: 14px;
      color: #333;
      cursor: pointer;
      outline: none;
      transition: border-color 0.2s;
    }

    #kbt-lang-select:hover {
      border-color: #2ecc71;
    }

    #kbt-lang-select:focus {
      border-color: #2ecc71;
      box-shadow: 0 0 0 2px rgba(46, 204, 113, 0.2);
    }
  `;
  shadow.appendChild(style);

  // 图片资源路径
  const iconTransUrl = chrome.runtime.getURL('src/icons/trans.png');
  const iconTransOverUrl = chrome.runtime.getURL('src/icons/trans_over.png');
  const iconSetUrl = chrome.runtime.getURL('src/icons/set.png');

  // 构建 UI 结构
  const container = document.createElement('div');
  container.className = 'kbt-container';
  
  // 强制内联样式，确保不受外部干扰
  container.style.cssText = `
    position: fixed !important;
    top: 50%;
    right: 0;
    transform: translateY(-50%);
    display: flex !important;
    flex-direction: column !important;
    gap: 15px !important;
    z-index: 2147483647 !important;
    margin: 0 !important;
    padding: 0 !important;
    width: auto !important;
    height: auto !important;
    background: transparent !important;
    border: none !important;
    box-shadow: none !important;
  `;

  // 1. 翻译按钮容器
  const transBtn = document.createElement('div');
  transBtn.className = 'kbt-icon-btn';
  transBtn.id = 'kbt-trans-btn';
  // 阻止原生拖拽
  transBtn.ondragstart = () => false;

  const transImg = document.createElement('img');
  transImg.src = iconTransUrl;
  transImg.className = 'kbt-icon-img';
  transImg.draggable = false;
  
  const loader = document.createElement('div');
  loader.className = 'loader kbt-hidden';

  transBtn.appendChild(transImg);
  transBtn.appendChild(loader);

  // 2. 设置按钮容器
  // 创建一个包裹容器来控制显示/隐藏区域
  const setBtnContainer = document.createElement('div');
  setBtnContainer.id = 'kbt-set-btn-container';

  const setBtn = document.createElement('div');
  setBtn.className = 'kbt-icon-btn';
  setBtn.id = 'kbt-set-btn';
  setBtn.ondragstart = () => false;

  const setImg = document.createElement('img');
  setImg.src = iconSetUrl;
  setImg.draggable = false;
  
  setBtn.appendChild(setImg);
  setBtnContainer.appendChild(setBtn);

  // 3. 设置面板 (初始隐藏)
  const settingsPanel = document.createElement('div');
  settingsPanel.className = 'kbt-settings-panel kbt-hidden';
  
  settingsPanel.innerHTML = `
    <div class="kbt-settings-header">
      <img src="${iconTransUrl}" class="kbt-header-icon" />
      <span>翻译设置 Translation Settings</span>
    </div>
    <div class="kbt-settings-body">
      <div class="kbt-form-group">
        <label>原文语言</label>
        <div class="kbt-fake-select">自动检测</div>
      </div>
      <div class="kbt-form-group">
        <label>译文语言</label>
        <select id="kbt-lang-select">
          ${SUPPORTED_LANGUAGES.map(lang => 
            `<option value="${lang.code}">${lang.name} (${lang.code})</option>`
          ).join('')}
        </select>
      </div>
    </div>
  `;

  // 组装 DOM
  container.appendChild(transBtn);
  container.appendChild(setBtnContainer); // 改为 append setBtnContainer
  container.appendChild(settingsPanel);
  shadow.appendChild(container);

  // --- 交互逻辑：Set 按钮滑入滑出 ---
  let hideTimer = null;

  function showSetBtn() {
    if (hideTimer) clearTimeout(hideTimer);
    container.classList.add('kbt-set-visible');
  }

  function hideSetBtn() {
    // 延迟隐藏，给鼠标移动留出时间
    hideTimer = setTimeout(() => {
      // 只有当设置面板未打开时才隐藏
      if (settingsPanel.classList.contains('kbt-hidden')) {
        container.classList.remove('kbt-set-visible');
      }
    }, 100);
  }

  // 鼠标移入 transBtn -> 显示 setBtn
  transBtn.addEventListener('mouseenter', showSetBtn);
  transBtn.addEventListener('mouseleave', hideSetBtn);

  // 鼠标移入 setBtn -> 保持显示
  setBtnContainer.addEventListener('mouseenter', showSetBtn);
  setBtnContainer.addEventListener('mouseleave', hideSetBtn);

  // 鼠标移入设置面板 -> 保持显示
  settingsPanel.addEventListener('mouseenter', showSetBtn);
  settingsPanel.addEventListener('mouseleave', hideSetBtn);

  // --- 拖拽交互逻辑 ---
  let isDragging = false;
  let startY = 0;
  let initialTop = 0;
  let hasMoved = false;

  // 绑定在 container 上
  container.addEventListener('mousedown', (e) => {
    // 只有点击在按钮上才允许拖拽
    if (!e.target.closest('.kbt-icon-btn')) return;
    // 设置面板内部不触发拖拽
    if (e.target.closest('.kbt-settings-panel')) return;

    isDragging = true;
    hasMoved = false;
    startY = e.clientY;
    
    // 获取当前 top 和 right/left
    const rect = container.getBoundingClientRect();
    const offsetX = e.clientX - rect.left;
    const offsetY = e.clientY - rect.top;

    // 改为 absolute 定位模式，便于自由拖拽
    // 注意：需要将 right/bottom 重置，完全依赖 left/top
    // 关键修复：移除 transform 以避免其对定位的干扰
    container.style.position = 'fixed';
    container.style.transform = 'none'; // 确保没有 translateY 干扰
    container.style.right = 'auto';
    container.style.bottom = 'auto';
    
    // 初始位置设为当前 rect 位置，避免跳动
    container.style.left = rect.left + 'px';
    container.style.top = rect.top + 'px';
    
    // 记录鼠标在容器内的偏移量，以便拖拽时保持相对位置
    container.dataset.offsetX = offsetX;
    container.dataset.offsetY = offsetY;
    
    document.body.style.userSelect = 'none';
  });

  document.addEventListener('mousemove', (e) => {
    if (!isDragging) return;
    
    const offsetX = parseFloat(container.dataset.offsetX || 0);
    const offsetY = parseFloat(container.dataset.offsetY || 0);

    const newLeft = e.clientX - offsetX;
    const newTop = e.clientY - offsetY;
    
    // 简单的阈值判断区分点击
    if (Math.abs(e.clientY - startY) > 2 || Math.abs(e.clientX - (newLeft + offsetX)) > 2) {
      hasMoved = true;
    }

    container.style.left = newLeft + 'px';
    container.style.top = newTop + 'px';
  });

  document.addEventListener('mouseup', (e) => {
    if (!isDragging) return;
    isDragging = false;
    document.body.style.userSelect = '';

    // 吸附逻辑：强制回到右侧
    // 保持当前的 top (垂直位置不变)
    // 强制 right: 0, left: auto
    const currentTop = container.style.top;
    
    container.style.left = 'auto';
    container.style.right = '0px';
    container.style.top = currentTop; // 保持垂直位置

    setTimeout(() => {
      hasMoved = false;
    }, 0);
  });

  // --- 事件绑定 ---

  // 1. 翻译按钮点击
  transBtn.addEventListener('click', (e) => {
    if (hasMoved) return; // 如果是拖拽，不触发点击
    
    if (isTranslating) {
      stopTranslation();
      return;
    }
    if (isTranslated) {
      resetState();
      return;
    }
    startTranslation();
  });

  // 2. 设置按钮点击
  setBtn.addEventListener('click', (e) => {
    if (hasMoved) return;
    e.stopPropagation();
    settingsPanel.classList.toggle('kbt-hidden');
  });

  // 3. 设置面板逻辑
  const langSelect = shadow.getElementById('kbt-lang-select');
  
  // 初始化值
  chrome.storage.local.get(['targetLang'], (result) => {
    if (result.targetLang) {
      langSelect.value = result.targetLang;
    }
  });

  langSelect.addEventListener('change', (e) => {
    const newLang = e.target.value;
    targetLang = newLang;
    chrome.storage.local.set({ targetLang: newLang }, () => {
      // console.log(`[跨境翻译宝] 目标语言已更新为: ${newLang}`);
    });
  });

  // 点击外部关闭
  document.addEventListener('click', (e) => {
    if (e.target !== host && !settingsPanel.classList.contains('kbt-hidden')) {
      settingsPanel.classList.add('kbt-hidden');
    }
  });

  // --- 核心翻译功能函数 (保留原有逻辑) ---

  function isVisible(el) {
    // 1. 基础检测
    if (!el.isConnected) return false;
    
    // 2. 样式检测
    const style = window.getComputedStyle(el);
    if (style.display === 'none' || style.visibility === 'hidden' || style.opacity === '0') {
      return false;
    }

    // 3. 尺寸检测 (放宽限制：有些文本节点的父级可能尺寸很小)
    const rect = el.getBoundingClientRect();
    if (rect.width === 0 && rect.height === 0) {
      // 特例：如果是 span 包裹的换行符等，可能宽高为0，但如果里面有可见文本，应该算可见吗？
      // 通常宽高为0意味着不可见。保持原逻辑，但注意 overflow。
      // 有些元素可能 overflow:hidden 且高度为0，确实不可见。
      return false;
    }

    // 4. 视口检测 (恢复视口限制，但增加缓冲区)
    // 之前的全页扫描导致了非视口区域的资源浪费。
    // 现在我们只翻译 Viewport + 上下 500px 缓冲区的内容。
    const viewHeight = window.innerHeight || document.documentElement.clientHeight;
    const buffer = 500;
    
    if (rect.bottom < -buffer || rect.top > viewHeight + buffer) {
       return false; // 在视口外
    }
    
    return true; 
  }

  function collectTextNodes(root = document.body) {
    // 关键修正：不再收集所有 TextNode，而是优先识别 "段落级" 元素，
    // 以避免将包含链接的句子拆分成多个片段。
    
    // 我们仍然需要一个 TreeWalker，但策略调整：
    // 如果遇到一个 Block 级元素，尝试获取其 innerText (作为整体翻译)
    // 但 innerText 包含子元素文本，直接替换 innerText 会破坏 HTML 结构 (如链接)。
    // 这是一个经典难题。
    
    // 妥协方案：
    // 保持 TextNode 遍历，但识别 "混合内容"。
    // 实际上，目前的拆分翻译是由于 TreeWalker 本质决定的。
    // 要实现 "原文(文本+链接)保持完整，译文独立"，
    // 意味着我们需要在父级 Block 元素末尾插入 整个 Block 的译文，
    // 而不是在每个 TextNode 后面插入。
    
    // 改进策略：
    // 1. 遍历 Block 级元素 (P, DIV, H1-H6, LI, TD)
    // 2. 检查其是否包含 "混合内容" (文本 + 链接/加粗等)
    // 3. 如果是混合内容，则提取纯文本进行翻译，并将译文 append 到 Block 末尾 (Block 模式)
    // 4. 如果是纯文本，维持原有逻辑。
    
    // 但这需要大幅重构遍历逻辑。
    // 鉴于时间，我们先采用一个 "分组" 策略：
    // 如果 TextNode 的父级是 A/B/SPAN/STRONG 等行内元素，
    // 我们向上查找最近的 Block 级祖先。
    // 我们以 Block 级祖先为单位进行翻译？
    
    // 风险：如果 Block 很大 (如整个文章容器)，整体翻译会很糟糕。
    // 只有 "段落级" Block (P, LI, Hx) 适合整体翻译。
    
    // 实施方案 v2.1:
    // 仅针对 P, H1-H6, LI, TD 标签，如果它们包含子元素 (A, SPAN等)，
    // 我们不单独收集其内部的 TextNode，而是标记该 Block 待翻译。
    // 但 TreeWalker 会深入子节点。
    
    // 让我们修正 collectTextNodes，使其能跳过已被 "Block级处理" 的子节点。
    
    const blockTags = ['P', 'H1', 'H2', 'H3', 'H4', 'H5', 'H6', 'LI', 'TD', 'TH', 'FIGCAPTION', 'DT', 'DD'];
    const nodes = [];
    
    // 第一步：查找所有可见的 Block 级目标
    // 我们只关心那些包含直接文本 或 包含行内标签的 Block
    
    // 为了不破坏现有逻辑的稳定性，我们微调：
    // 如果一个 TextNode 的父级是行内元素 (A, SPAN...)，
    // 且该行内元素的父级是 段落级 Block (P, LI...)，
    // 且该 Block 尚未被处理。
    // 我们记录该 Block，并将其 innerText 作为一个整体加入翻译队列。
    // 翻译完成后，我们在该 Block 末尾追加译文 DIV。
    // 这样原文结构（含链接）保持不变。
    
    // 但这会导致：Block 内的部分文本被翻译，部分未翻译？
    // 不，我们应该跳过 Block 内的所有 TextNode。
    
    const walker = document.createTreeWalker(
      root,
      NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_TEXT,
      {
        acceptNode: (node) => {
          if (processedNodes.has(node)) return NodeFilter.FILTER_REJECT;
          
          if (node.nodeType === Node.ELEMENT_NODE) {
            const tag = node.tagName;
            // 排除非翻译容器
            if (['SCRIPT', 'STYLE', 'NOSCRIPT', 'TEXTAREA', 'INPUT', 'CODE', 'PRE', 'SVG', 'IMG'].includes(tag)) {
              return NodeFilter.FILTER_REJECT;
            }
            // 检查是否为目标 Block
            if (blockTags.includes(tag)) {
               return NodeFilter.FILTER_ACCEPT; // 选中 Block
            }
            return NodeFilter.FILTER_SKIP; // 继续遍历子节点
          }
          
          if (node.nodeType === Node.TEXT_NODE) {
             // 如果文本节点的父级是 Block (且该 Block 没被选中，可能是 DIV)，或者父级是行内元素
             // 这里逻辑比较复杂，为了简化：
             // 我们只抓取 Block 元素，和 游离的 TextNode (父级不是上述 Block)
             return NodeFilter.FILTER_ACCEPT;
          }
          
          return NodeFilter.FILTER_SKIP;
        }
      }
    );

    // 实际执行比较困难，回退到更稳健的方案：
    // 保持现有的 TextNode 收集，
    // 但在 applyTranslationToDOM 时，
    // 如果发现 TextNode 是某个 "混合段落" 的一部分，
    // 我们不要原地插入，而是尝试合并？
    
    // 不，用户明确要求 "原文(文本和链接)保持完整，译文是独立的"。
    // 这意味着必须以 Block 为单位翻译。
    
    // 重新实现 collectTextNodes：
    // 优先收集 Block 元素 (P, LI, Hx)。
    // 如果 Block 元素包含直接文本，则视为一个翻译单元。
    // 收集后，标记该 Block 为 processed，其内部所有子节点不再重复收集。
    
    // 新的遍历器：只看 Element，如果是 Block 且可见，加入 list。
    // 如果是 DIV，继续遍历。
    // 如果是 TextNode (且未被 Block 包含)，加入 list。
    
    // 这需要手动递归，TreeWalker 不好控制 "包含关系"。
    
    function traverse(node) {
      if (processedNodes.has(node)) return;
      
      // Handle Element
      if (node.nodeType === Node.ELEMENT_NODE) {
         if (!isVisible(node)) return; 
         const tag = node.tagName;
         if (['SCRIPT', 'STYLE', 'NOSCRIPT', 'TEXTAREA', 'INPUT', 'CODE', 'PRE', 'SVG', 'IMG'].includes(tag)) return;
         
         // Shadow DOM support
         if (node.shadowRoot) {
            Array.from(node.shadowRoot.childNodes).forEach(traverse);
         }
         
         // Atomic Block Logic
         const isAtomicBlock = blockTags.includes(tag) && !hasBlockChildren(node);
         if (isAtomicBlock) {
            // 修复：仅提取可见文本，避免获取 aria-label, hidden text 等干扰内容
            // node.innerText 虽然通常只包含渲染文本，但有时会包含 display:none 的伪元素内容或被 CSS 隐藏的内容
            // 更重要的是，innerText 会包含所有子节点的文本。
            
            // 为了过滤掉像 Material Icon "style" 这种虽然是文本但其实是图标的内容，
            // 或者隐藏的辅助文本。
            
            // 这里我们手动构建 text，只包含可见的 TextNode 子节点
            let pureText = '';
            // 使用 TreeWalker 仅遍历 TextNode
            const walker = document.createTreeWalker(node, NodeFilter.SHOW_TEXT, {
               acceptNode: (n) => {
                  // 排除父级是不可见的
                  if (!isVisible(n.parentElement)) return NodeFilter.FILTER_REJECT;
                  // 排除 Material Icons 等图标字体
                  // 简单判断：如果父级 class 包含 material-icons 等常见图标库
                  const parentClass = n.parentElement.className;
                  if (typeof parentClass === 'string' && 
                      (parentClass.includes('material-icons') || 
                       parentClass.includes('fa-') || 
                       parentClass.includes('icon'))) {
                      return NodeFilter.FILTER_REJECT;
                  }
                  return NodeFilter.FILTER_ACCEPT;
               }
            });
            
            while(walker.nextNode()) {
               pureText += walker.currentNode.textContent;
            }
            
            pureText = pureText.trim();

            // 放宽正则：只要不是纯数字/符号即可 (支持英文、拉丁语系等)
            // 之前的正则 /[a-zA-Z\u4e00-\u9fa5]/ 可能遗漏了某些语言
            // 新正则：排除纯数字、纯标点
            if (pureText && /[^\d\s\u2000-\u206F\u2E00-\u2E7F\\'!"#$%&()*+,\-.\/:;<=>?@\[\]^`{|}~]/.test(pureText)) {
               nodes.push({ type: 'block', node: node, text: pureText });
               processedNodes.add(node);
               markDescendantsProcessed(node);
            }
            return;
         }
         
         // Traverse Light DOM children
         Array.from(node.childNodes).forEach(traverse);
      }
      
      // Handle Text Node
      else if (node.nodeType === Node.TEXT_NODE) {
         const text = node.textContent.trim();
         // 同上放宽正则
         if (text && /[^\d\s\u2000-\u206F\u2E00-\u2E7F\\'!"#$%&()*+,\-.\/:;<=>?@\[\]^`{|}~]/.test(text)) {
            nodes.push({ type: 'text', node: node, text });
            processedNodes.add(node);
         }
      }
    }
    
    function hasBlockChildren(el) {
       return Array.from(el.children).some(c => blockTags.includes(c.tagName) || c.tagName === 'DIV');
    }
    
    function markDescendantsProcessed(el) {
       const w = document.createTreeWalker(el, NodeFilter.SHOW_ALL);
       while(w.nextNode()) processedNodes.add(w.currentNode);
    }
    
    traverse(root);
    // console.log(`[跨境翻译宝] 扫描完成，共找到 ${nodes.length} 个待翻译节点`); // 发布时注释掉调试日志
    return nodes;
  }

  // --- 动态翻译监听器 ---
  let scrollObserver = null;
  let mutationObserver = null;
  let scrollTimer = null;
  let mutationTimer = null;

  // 1. 滚动监听 (Debounce 200ms)
  function setupScrollObserver() {
    const handleScroll = () => {
      if (scrollTimer) clearTimeout(scrollTimer);
      scrollTimer = setTimeout(() => {
        if (!isTranslating && !isTranslated) return;
        // 触发增量翻译
        // 复用 startTranslation 的核心逻辑，但需要调整一下
        // 我们抽取一个 coreTranslate 函数
        continueTranslation();
      }, 200);
    };
    
    window.addEventListener('scroll', handleScroll, { passive: true });
    return () => window.removeEventListener('scroll', handleScroll);
  }

  // 2. DOM 变动监听 (Debounce 500ms)
  function setupMutationObserver() {
    const observer = new MutationObserver((mutations) => {
      let hasMeaningfulChanges = false;
      for (const mutation of mutations) {
        if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
           // 增强过滤：
           // 1. 忽略 SCRIPT/STYLE/LINK 等非可视元素的插入
           // 2. 忽略 kbt 自身的元素
           
           for (const node of mutation.addedNodes) {
              if (node.nodeType === Node.ELEMENT_NODE) {
                 const tag = node.tagName;
                 if (['SCRIPT', 'STYLE', 'LINK', 'META', 'NOSCRIPT'].includes(tag)) continue;
                 if (node.id === 'kbt-root' || node.classList?.contains('kbt-inline-loader') || node.classList?.contains('kbt-translation-block')) continue;
                 
                 // 检查新元素是否可见 (简单检查，isVisible 开销较大，留给 collectTextNodes 做)
                 // 但这里可以先排除一些明显的隐藏元素
                 
                 hasMeaningfulChanges = true;
                 break;
              } else if (node.nodeType === Node.TEXT_NODE && node.textContent.trim()) {
                 hasMeaningfulChanges = true;
                 break;
              }
           }
           if (hasMeaningfulChanges) break;
        }
      // 2. 监听属性变化 (针对隐藏->显示)
      // 许多下拉框/Tooltip 早就存在于 DOM 中，只是通过 display/visibility/opacity 切换
      if (mutation.type === 'attributes' && 
          (mutation.attributeName === 'style' || mutation.attributeName === 'class')) {
          
          const target = mutation.target;
          if (target.nodeType === Node.ELEMENT_NODE) {
             // 检查是否变为可见
             // 我们不需要极其精确，只要它现在是可见的，且之前可能没被翻译（processedNodes 检查在 continueTranslation 中做）
             // 只有当它包含文本时才值得触发
             if (isVisible(target) && target.innerText.trim().length > 0) {
                hasMeaningfulChanges = true;
                break;
             }
          }
      }
    }

    if (hasMeaningfulChanges) {
        if (mutationTimer) clearTimeout(mutationTimer);
        mutationTimer = setTimeout(() => {
          if (!isTranslating && !isTranslated) return;
          continueTranslation();
        }, 500);
      }
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true,
      attributes: true, // 开启属性监听
      attributeFilter: ['style', 'class', 'hidden'] // 仅监听可能影响可见性的属性
    });
    
    return observer;
  }

  // 增量翻译入口
  async function continueTranslation() {
    // 类似于 startTranslation，但只处理未处理的节点
    // 由于 collectTextNodes 内部已经有 processedNodes 检查，
    // 直接调用它只会返回新节点。
    
    // 如果当前正在进行网络请求，是否需要排队？
    // processBatch 是 async 的。
    // 我们简单处理：如果正在翻译中（isTranslating），可以继续追加。
    // 但如果用户已经点击完成（isTranslated），说明是增量更新。
    
    // 无论哪种状态，只要还没 Reset，就可以继续翻译新内容。
    if (!isTranslating && !isTranslated) return;

    try {
      const textNodes = collectTextNodes();
      if (textNodes.length === 0) return;

      // 如果发现新节点，且 UI 显示已完成，则切换回 Loading 状态？
      // 或者静默更新？
      // 为了体验，如果是滚动加载，最好静默一点，不要频繁切换图标。
      // 但如果是弹窗，用户需要反馈。
      // 保持 Inline Loader 即可，不需要切换主图标状态。

      let currentBatch = [];
      let currentLength = 0;

      for (const item of textNodes) {
        const { node, text, type } = item;
        
        // 插入 Inline Loader
        const inlineLoader = document.createElement('span');
        inlineLoader.className = 'kbt-inline-loader';
        
        try {
          if (type === 'block') {
             // 修复：检查 node 是否仍然连接在 DOM 中
             if (node.isConnected) {
                node.appendChild(inlineLoader);
             }
          } else {
             // 修复：检查 node.parentElement 是否存在
             if (node.parentElement && node.parentElement.isConnected) {
               if (node.nextSibling) {
                 node.parentElement.insertBefore(inlineLoader, node.nextSibling);
               } else {
                 node.parentElement.appendChild(inlineLoader);
               }
             } else {
               // 如果父节点已断开连接，跳过此节点
               continue;
             }
          }
        } catch (domError) {
          // 忽略 DOM 操作错误，继续处理下一个节点
          console.warn('[跨境翻译宝] DOM 操作失败 (已忽略):', domError);
          continue; 
        }
        
        inlineLoaders.set(node, inlineLoader);

        if (currentLength + text.length > MAX_BATCH_SIZE && currentBatch.length > 0) {
          await processBatch(currentBatch, abortController ? abortController.signal : null);
          currentBatch = [];
          currentLength = 0;
        }

        currentBatch.push(item);
        currentLength += text.length;
        processedNodes.add(node);
      }

      if (currentBatch.length > 0) {
        // 如果 abortController 已被置空（说明停止了），则不继续
        // 但这里我们允许增量翻译，所以需要确保 abortController 存在
        if (!abortController) abortController = new AbortController();
        await processBatch(currentBatch, abortController.signal);
      }

    } catch (error) {
      if (error.name !== 'AbortError') {
        console.error('[跨境翻译宝] 增量翻译出错:', error);
      }
    }
  }

  // --- 修改 collectTextNodes 调用处 ---
  // 需要调整 startTranslation 和 processBatch 以适应新的 item 结构 {type, node, text}


  async function startTranslation() {
    isTranslating = true;
    abortController = new AbortController();
    
    transImg.classList.add('kbt-hidden');
    loader.classList.remove('kbt-hidden');
    
    // 启动监听器
    if (!scrollObserver) scrollObserver = setupScrollObserver();
    if (!mutationObserver) mutationObserver = setupMutationObserver();

    try {
      // 复用 continueTranslation 逻辑
      // 但初次需要触发完成状态
      await continueTranslation();
      
      // 初次全量扫描完成后，进入“已翻译”状态，但监听器保持开启
      finishTranslation();

    } catch (error) {
      if (error.name !== 'AbortError') {
        console.error('[跨境翻译宝] 翻译出错:', error);
        alert('翻译服务暂时不可用');
      }
      stopTranslation();
    }
  }

  async function processBatch(batch, signal) {
    const separator = '\n\n'; 
    const fullText = batch.map(item => item.text).join(separator);

    const response = await fetch(WORKER_URL, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        text: fullText,
        source_lang: 'auto',
        target_lang: targetLang
      }),
      signal
    });

    if (!response.ok) throw new Error(`API Error: ${response.status}`);

    const data = await response.json();
    const translatedSegments = data.translation.split(separator);

    batch.forEach((item, index) => {
      if (translatedSegments[index]) {
        applyTranslationToDOM(item, translatedSegments[index]);
      }
    });
  }

  function applyTranslationToDOM(item, translatedText) {
    const { node, type, text: originalText } = item;

    // 移除对应的 Inline Loader
    const loader = inlineLoaders.get(node);
    if (loader) {
      loader.remove();
      inlineLoaders.delete(node);
    }

    if (type === 'block') {
       // --- Block 模式处理 ---
       // 原文完整保留，译文追加在 Block 末尾
       const transContainer = document.createElement('div');
       transContainer.style.color = '#333'; 
       transContainer.style.fontWeight = 'normal'; // 取消加粗
       transContainer.style.marginTop = '6px';
       transContainer.style.fontSize = '0.95em';
       transContainer.style.lineHeight = '1.5';
       transContainer.className = 'kbt-translation-block';
       transContainer.textContent = translatedText;
       
       node.appendChild(transContainer);
       
    } else {
       // --- Text 模式处理 (游离文本) ---
       const parent = node.parentElement;
       const parentTag = parent.tagName;
       
       // 强制同行的标签
       const forceInlineTags = ['A', 'SPAN', 'B', 'STRONG', 'I', 'EM', 'SMALL', 'LABEL', 'BUTTON', 'FIGCAPTION'];
       const shouldBlockDisplay = !forceInlineTags.includes(parentTag) && (originalText.length > 50 || originalText.includes('\n'));
    
       if (!shouldBlockDisplay) {
          // 同行显示
          modifiedTextNodes.set(node, originalText);
          node.textContent = `${originalText} ${translatedText}`;
       } else {
          // 换行显示
          const transContainer = document.createElement('div');
          transContainer.style.color = '#333'; 
          transContainer.style.fontWeight = 'normal'; // 取消加粗
          transContainer.style.marginTop = '4px';
          transContainer.style.fontSize = '0.95em';
          transContainer.style.lineHeight = '1.5';
          transContainer.className = 'kbt-translation-block';
          transContainer.textContent = translatedText;
    
          if (node.nextSibling) {
            parent.insertBefore(transContainer, node.nextSibling);
          } else {
            parent.appendChild(transContainer);
          }
       }
    }
  }

  function stopTranslation() {
    if (abortController) {
      abortController.abort();
      abortController = null;
    }
    // 销毁监听器
    if (scrollObserver) {
      scrollObserver(); // 执行返回的 cleanup 函数
      scrollObserver = null;
    }
    if (mutationObserver) {
      mutationObserver.disconnect();
      mutationObserver = null;
    }
    if (scrollTimer) clearTimeout(scrollTimer);
    if (mutationTimer) clearTimeout(mutationTimer);
    
    resetState();
  }

  function finishTranslation() {
    // 保持监听器活跃，以便支持滚动和动态内容
    isTranslating = false;
    isTranslated = true;
    loader.classList.add('kbt-hidden');
    transImg.classList.remove('kbt-hidden');
    transImg.src = iconTransOverUrl;
  }

  function resetState() {
    isTranslating = false;
    isTranslated = false;
    
    // 如果是 reset (手动重置)，也需要确保监听器被移除
    // 但通常 resetState 是被 stopTranslation 调用的
    // 如果直接调用 resetState (例如从已完成状态点击重置)
    if (scrollObserver) {
      scrollObserver(); 
      scrollObserver = null;
    }
    if (mutationObserver) {
      mutationObserver.disconnect();
      mutationObserver = null;
    }
    
    // 1. 移除所有翻译插入的 Block 元素
    document.querySelectorAll('.kbt-translation-block').forEach(el => el.remove());
    
    // 2. 还原被修改的短语文本
    modifiedTextNodes.forEach((originalText, node) => {
      // 检查节点是否仍在 DOM 中（虽然 WeakSet 自动处理引用，但 Map 强引用，需注意）
      // 简单直接还原
      node.textContent = originalText;
    });
    modifiedTextNodes.clear();

    // 3. 移除所有剩余的 Inline Loaders (如果是在翻译中途停止)
    inlineLoaders.forEach((loader) => loader.remove());
    inlineLoaders.clear();

    // 4. 重置已处理节点集合
    processedNodes = new WeakSet();

    loader.classList.add('kbt-hidden');
    transImg.classList.remove('kbt-hidden');
    transImg.src = iconTransUrl;
  }

})();
